/*
 * Declaration of PenguinPlay exception types
 * #included by PenguinPlay.h - please do not include directly
 *
 * License: See the accompanying file LICENSE
 *
 * Last change by $Author$
 * on             $Date$
 * to             $Revision$
 *
 */


#ifndef _PP_EXCEPTIONS_H
#define _PP_EXCEPTIONS_H 1


//#include <PenguinPlay/PenguinPlay.h>

// using namespace std; // FIX: Why that???


/*
 * Some notes about Exception usage:
 *
 * There are three basic situations where Exceptions are thrown:
 *
 * * Exceptions used for debugging. Those should only be caught by some
 *   toplevel error message dumper, exiting the app. Those should include as
 *   much info as possible, including __LINE__ etc, but they also have to be
 *   used only while debugging. Use the ppThrow (description) macro for this.
 *
 * * Exceptions indicating fatal errors that are not handled by the lib.
 *   Checks that can't be made obsolete by debugging, errors that have to be
 *   reported to the calling app. Mostly caught by that app. Here detailed
 *   info isn't really important, but some human-readable info should be there
 *   in case the app does *not* handle it. Throw the Exceptions with the
 *   "origin" and "details" options supplied for this.
 *
 * * Exceptions that *are* handled by the lib internally. Here textual info
 *   is completely unneccessary. Throw the Exceptions without parameters for
 *   this.
 */

//
// Base exception class.
// Later we might feel the need to make this thing a bit cleverer.
// Update: I tried to improve it - hope this is fine (update from creinig)
//
class ppException
{
private:

	const char *_Origin;  // In which method was it thrown?
	const char *_Type;    // What exception category is it in?
	const char *_Details; // What exactly happened?
	const char *_File;    // Source file where it originated
	const char *_Line;    // Source line where it originated


public:

	// generic constructor
	ppException (void) :
		_Origin  (""),
		_Type    (""),
		_Details (""),
		_File    (""),
		_Line    ("")
	{}

	// Constructor for custom setting of the fields. Usually called
	// in the initializer list of derived classes
	ppException (const char *origin, const char *type, const char *details,
		     const char *file, const char *line) :
	        _Origin  (origin),
		_Type    (type),
		_Details (details),
		_File    (file),
		_Line    (line)
	{}



	// Every error of given type has a particular.
	const char *Type    (void) { return _Type;    }

	// These strings should be set on a per construction basis.
	// The constructors of derived exception classes need to set them:
	// ppSomeException::ppSomeException (const char *Org, const char *Det);
	const char *Origin  (void) { return _Origin;  };
	const char *Details (void) { return _Details; };
	const char *File    (void) { return _File;    };
	const char *Line    (void) { return _Line;    };

  //FIX we should perhaps have a "operator<<(ostream&)" function or the like
  //but the corporate will has not yet decided how we handle this.
  //(i.e. we haven't decided between iostream or stdio).
};




// One or more function/method parameter(s) had an invalid value
class ppEInvalidParameter : public ppException
{
public:
	ppEInvalidParameter (void) {}

	ppEInvalidParameter (const char *origin, const char *details,
			     const char *file = "", const char *line = "") :
		ppException (origin,
			     "One parameter had an invalid value",
			     details, file, line)
	{}

};



// For paranoia tests. Should only be thrown when the code/compiler/OS is buggy
class ppEObscureFailure : public ppException
{
public:
	ppEObscureFailure (void) {}

	ppEObscureFailure (const char *origin, const char *details,
			   const char *file = "", const char *line = "") :
		ppException (origin,
			     "This should never happen. Please"
			     "send the maintainers a note about this incident",
			     details, file, line)
	{}
};



// Thrown when some aspect of the environment is not supported or a required
// "feature" is not present. Note: this is not for checks that can be done at
// compile time.
class ppEUnsupportedEnvironment : public ppException
{
public:
	ppEUnsupportedEnvironment (void) {}

	ppEUnsupportedEnvironment (const char *origin, const char *details,
				   const char *file = "", const char *line = "") :
		ppException (origin,
			     "This environment is not supported (yet). Please"
			     "send the maintainers a note about this incident",
			     details, file, line)
	{}
};



// Thrown e.g. when decoding corrupt files etc
class ppEInvalidData : public ppException
{
public:
	ppEInvalidData (void) {}

	ppEInvalidData (const char *origin, const char *details,
			const char *file = "", const char *line = "") :
		ppException (origin,
			     "The data currently processed is invalid",
			     details, file, line)
	{}
};



// Service not available in this particular case.
class ppEUnavailable : public ppException
{
public:
	ppEUnavailable (void) {}

	ppEUnavailable (const char *origin, const char *details,
			  const char *file = "", const char *line = "") :
		ppException (origin,
			     "Service unavailable",
			     details, file, line)
	{}
};


// File opening / memory allocation / visual initialization / ... failed
class ppEAccessFailure : public ppException
{
public:
  ppEAccessFailure (const char *origin, const char *details,
                   const char *file = "", const char *line = "") :
    ppException (origin,
                 "Attempt to access resource failed",
                 details, file, line)
    {}
};


// Assertion failed
// FIX: Not used yet. ppAssert () still yields a ppInternalError ()
class ppEAssertionFailed : public ppException
{
public:
	ppEAssertionFailed (void) {}

	ppEAssertionFailed (const char *origin, const char *details,
			    const char *file = "", const char *line = "") :
		ppException (origin,
			     "An Assertion failed",
			     details, file, line)
	{}
};



// Error occured while initializing an object
// FIX: I'm not sure if we should use this one
class ppEConstructionFailed : public ppException
{
public:
	ppEConstructionFailed (void) {}

	ppEConstructionFailed (const char *origin, const char *details,
			       const char *file = "", const char *line = "") :
		ppException (origin,
			     "An error occured during object initialization",
			     details, file, line)
	{}
};



// Use when nothing else matches
class ppEMethodFailure : public ppException
{
public:
	ppEMethodFailure (void) {}

	ppEMethodFailure (const char *origin, const char *details,
			  const char *file = "", const char *line = "") :
	        ppException (origin,
			     "Method failed",
			     details, file, line)
	{}
};




#endif
