#include <PenguinPlay/PenguinPlay.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include <string> // STL string
#include <new>

#include <PenguinPlay/Codec.h>
#include <PenguinPlay/SampleIff.h>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//=============================================================================
// IFF - Interchange File Format
//=============================================================================
// CHUNK ID's
enum CKID_IFF
{
  CKID_IFF_NAME =(0x454d414e),    // "NAME"
  CKID_IFF_VHDR =(0x52444856),    // "VHDR"
  CKID_IFF_BODY =(0x59444f42),    // "BODY"
  CKID_IFF_ANNO =(0x4f4e4e41),    // "ANNO"
  CKID_IFF_COPY =(0x20294328),    // "(C) "
  CKID_IFF_AUTH =(0x48545541)     // "AUTH"
};

#pragma pack(1)
struct SampleIff::Voice8Header          // size = 20 = 0x14
{
  unsigned long  OneShotHi;
  unsigned long  RepeatHi;
  unsigned long  SmpPerHiCycle;
  unsigned short SmpPerSec;
  unsigned char  NumOctaves;
  unsigned char  Cmp;           // Cmp == 0 for no compression
  unsigned long  Volume;        // long OR short ?
};
#pragma pack()

//============================================================================
// constructors and destructors
//============================================================================
SampleIff::SampleIff()
{
}
SampleIff::~SampleIff()
{
}

//============================================================================
// SampleIff::load
//----------------------------------------------------------------------------
// Supports 8SVX 16SV (used in FT2)
// It also supports PCM, fibonacci DPCM, and exponential DPCM
//============================================================================
bool SampleIff::Load(const char *filename)
{
  Voice8Header  Header;
  struct stat   filestat; // used to get the length of the file
  long          Len;
  unsigned long chunkid;
  unsigned long chunksize;
  int           nextseek; // used to get the next chunk
  FILE*         fp = fopen(filename, "rb");

  try
    {
      if (fp == 0)
        {
          ppThrow (EAccessFailure, "error opening file");
        }

      if (fread(&chunkid, 4, 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "read error");
        }
      if (memcmp(&chunkid, "FORM",4))
        {
          ppThrow (EInvalidData, "FORM not found.");
        }
      if (fread(&Len, 4, 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "read error");
        }

      if (fread(&chunkid, 1, 4, fp) != 4)
        {
          ppThrow (EAccessFailure, "read error");
        }
      if (!memcmp(&chunkid, "8SVX", 4))
        {
         Set8Bit();
        }
      else if (!memcmp(&chunkid, "16SV",4))
        {
          Set16Bit();
          SetSigned();;
        }
      else
        { // UNKNOWN IFF SAMPLE FORMAT
          ppThrow (EUnsupported, "Unknown IFF sample format");
        }

      // READ IN CHUNKS IN ANY ORDER
      nextseek = ftell(fp);
      //  Len = ltmp + nextseek - 4;
      // DONT TRUST THE FILESIZE GIVEN IN THE FILE !!!!!!!
      stat(filename, &filestat);
      Len = filestat.st_size;
      while (nextseek < Len)
        {
          // SEEK TO NEXT CHUNK
          if (fseek(fp, nextseek, SEEK_SET))
            ppThrow (EAccessFailure, "seek error");

          // READ CHUNK ID
          if (fread(&chunkid, 1, 4, fp) != 4)
            {
              ppThrow (EAccessFailure, "read error");
            }
          // READ CHUNK SIZE
          if (fread(&chunksize, 4, 1, fp) != 1)
            ppThrow (EAccessFailure, "fread error");
          long ltmp = Codec::BE32(chunksize);
          chunksize = ltmp + (ltmp % 2); // pad if not word aligned
          nextseek = chunksize + ftell(fp);

          switch (chunkid)
            {
            case CKID_IFF_NAME: // "NAME"
              {
                char* name = new char [chunksize + 1];
                if (!name)
                  {
                    throw  std::bad_alloc ();
                  }
                if (fread(name, 1,chunksize, fp) != chunksize)
                  {
                    ppThrow (EAccessFailure, "read error");
                  }
                name[chunksize] = 0; // null terminate the  std::string
                SetName(name);
                delete [] name;
              }
              break;

            case CKID_IFF_VHDR: // "VHDR"
              if (fread(&Header, sizeof(Voice8Header), 1, fp) != 1)
                {
                  ppThrow (EAccessFailure, "read error");
                }
              // BODY SHOULD COME AFTER VHDR  - but it doesn't always :)
              break;

            case CKID_IFF_BODY: // "BODY"
              {
                m_length = chunksize;
                nextseek = ftell(fp) + m_length + (m_length % 2);

                if (Header.Cmp)
                  m_length >>=1;

                unsigned char* temp_data = new unsigned char [m_length];
                if (!temp_data)
                  {
                    throw  std::bad_alloc ();
                  }
                // READ IN SAMPLE DATA
                if (fread(temp_data, m_length, 1, fp) != 1)
                  {
                    if (ferror(fp))
                      {
                        ppThrow (EAccessFailure, "read error");
                      }
                  }
                switch (Header.Cmp)
                  {
                  case 0: // PCM
                    m_data0 = temp_data;
                    break;

                  case 1: // Fibonacci coding
                    m_data0 = new unsigned char [m_length*2];
                    if (!m_data0)
                      {
                        throw  std::bad_alloc ();
                      }
                    Codec::DecodeFibonacciDPCM(temp_data,
                                                  m_data0,
                                                  m_length);
                    m_length <<= 1;
                    delete [] temp_data;
                    break;

                  case 2: // exponential coding
                    m_data0 = new unsigned char [m_length*2];
                    if (!m_data0)
                      {
                        throw  std::bad_alloc ();
                      }
                    Codec::DecodeExponentialDPCM(temp_data,
                                                    m_data0,
                                                    m_length);
                    m_length <<= 1;
                    delete [] temp_data;
                    break;
                  }
                break;
              }

            default:
              ppWarning ("  Unknown Chunk : %4s 0x%lX\n",
                         (char *)&chunkid,
                         chunkid);
            } // END SWITCH
        } // END WHILE

      if (Header.Volume)
        m_volume    = Codec::BE32(Header.Volume) / 1024;
      if (Header.SmpPerSec != 0)
        {
          m_frequency = Codec::BE16(Header.SmpPerSec);
        }
      else
        {
          if (Is16Bit())
            m_frequency = 22048;
          else
            m_frequency = 11024;
        }

      // convert 8bit data to unsigned
      if (Is8Bit())
        for (Sample::position_t i = 0; i < m_length; ++i)
          m_data0[i] ^= 0x80;

      fclose(fp);
      return true;
    }
  catch ( Exception &message)
    {
      if (fp != 0) fclose(fp);
      throw message;
    }
}


//============================================================================
// SampleIff::Save
//----------------------------------------------------------------------------
//
//============================================================================
bool SampleIff::Save(const char *filename, const Sample& sample) const
{
  Voice8Header  Header;
  unsigned long chunksize;
  FILE*         fp = fopen(filename, "wb");

  try
    {
      if (fp == NULL)
        {
          ppThrow (EAccessFailure, "error opening file");
        }
      // FORM
      // leave space for filesize and come back later to write it
      if (fwrite("FORM    ", 8,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // LENGTH
      // CHUNK TYPE - 16SV or 8SVX
      if (sample.Is16Bit())
        {
          if (fwrite("16SV", 4, 1, fp) != 1)
            {
              ppThrow (EAccessFailure, "write error");
            }
        }
      else if (fwrite("8SVX", 4, 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }

      // NAME
      if (fwrite("NAME",4,1,fp)!= 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      chunksize = 22;
      if (fwrite(&chunksize, 4,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      if (fwrite(sample.GetName().c_str(), 22,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }

      // VHDR
      if (fwrite("VHDR", 4,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }

      chunksize = sizeof(Voice8Header);

      if (fwrite(&chunksize, sizeof(chunksize),1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // no of samples in the section that dont repeat
      Header.OneShotHi     = Codec::BE32(sample.GetLength());
      Header.RepeatHi      = 0;     // Sample->LoopEnd - Sample->LoopStart
      Header.SmpPerHiCycle = 0;
      Header.SmpPerSec     = Codec::BE32(sample.GetFrequency());
      Header.NumOctaves    = 0;
      Header.Cmp           = 0;     // 0 for no compression
      Header.Volume        = 65535; // full volume
      if (fwrite(&Header, sizeof(Voice8Header), 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // BODY
      // write id
      if (fwrite("BODY", 4,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
          return 0;
        }
      // write chunk size
      chunksize = sample.GetLength();
      if (fwrite(&chunksize, sizeof(chunksize),1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // convert 8bit data to signed
      if (!sample.Is16Bit())
        {
          for (Sample::position_t i = 0; i < sample.GetLength(); ++i)
            {
              sample.GetData()[i] ^= 0x80;
            }
        }

      if (fwrite(sample.GetData(), chunksize, 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // convert 8bit data back to unsigned -- this is could be done better
      if (!sample.Is16Bit())
        {
          for (Sample::position_t i = 0; i < sample.GetLength(); ++i)
            {
              sample.GetData()[i] ^= 0x80;
            }
        }

      // go back and write the size of the file
      long size = ftell(fp);
      fseek(fp, 4, SEEK_SET);
      fwrite(&size, 4, 1, fp);
      return true;

    }
  catch ( Exception &message)
    {
      fclose(fp);
      throw message;
    }
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
