#include <PenguinPlay/PenguinPlay.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef PP_HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/stat.h>

#include <string>

#include <PenguinPlay/Codec.h>
#include <PenguinPlay/SampleAu.h>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//=============================================================================
//  AU FILE FORMAT
//=============================================================================

// this header follows the magic id ".snd"
struct SampleAu::AU_HEADER
{
  unsigned long header_size;
  unsigned long length;        /* length in bytes */
  unsigned long format;
  unsigned long sample_rate;
  unsigned long channels;
};


//============================================================================
// constructors and destructors
//============================================================================
SampleAu::SampleAu()
{
}

SampleAu::~SampleAu()
{
}

//============================================================================
// SampleAu::Load
//----------------------------------------------------------------------------
// This supports :
//     8 bit mu-law
//     8 bit A-law
//     8 bit linear pcm
//    16 bit linear pcm
//============================================================================
bool SampleAu::Load(const char *filename)
{
  AU_HEADER      header;
  struct stat    filestat; /* used to get the length of the file */
  unsigned long  magic;
  unsigned char* temp_data = 0;
  unsigned int   i;

  FILE*          fp        = fopen(filename, "rb");

  try
    {

      if (fp == 0)
        {
          ppThrow (EAccessFailure, "error opening file\n");
        }

      stat(filename, &filestat);
      m_length = filestat.st_size;

      if (fread(&magic,4,1,fp)!=1)
        {
          ppThrow (EAccessFailure, "read error");
        }

      SetMono(); // usually mono

      if (!memcmp(&magic, ".snd", 4)) // header is present
        {
          if (fread(&header,sizeof(AU_HEADER),1,fp)!=1)
            {
              ppThrow (EAccessFailure, "read error");
            }
          m_frequency = Codec::BE32(header.sample_rate);
          m_length   -= Codec::BE32(header.header_size);
          temp_data   = new unsigned char [m_length];

          if (fread(temp_data, m_length, 1, fp)!=1)
            {
              ppThrow (EAccessFailure, "read error");
            }
          switch (Codec::BE32(header.format))
            {
            case 1: //8 bit mu-law G.711
              m_data0 = new unsigned char [GetLength()<<1];

              Codec::DecodeMuLaw(temp_data,
                                    (unsigned short*&)m_data0,
                                    GetLength());
              // multiply length by two (as samples are 16 bit)
              m_length <<=1;
              Set16Bit();
              delete temp_data;
              temp_data = 0;
              return true;

            case 2: // 8-bit linear
              m_data0 = temp_data;
              // convert to unsigned
              for (i = 0; i < m_length; ++i)
                m_data0[i] ^= 0x80;
              Set8Bit();
              return true;

            case 3: // 16-bit linear
              m_data0 = temp_data;
              Set16Bit();
              return true;

            case 27: // 8-bit A-Law G.71
              m_data0 = new unsigned char [m_length<<1];

              Codec::DecodeALaw(temp_data,
                                   (unsigned short *&)m_data0,
                                   m_length);
              // multiply length by two (as samples are 16 bit)
              m_length <<=1;

              Set16Bit();
              delete temp_data;
              temp_data = 0;
              return true;

            case 4: // 24-bit linear
            case 5: // 32-bit linear
            case 6: // Floating-point
            case 7: // Double precision floating point
            case 8: // Fragmented sample data
            case 10: // DSP program
            case 11: // 8-bit fixed point
            case 12: // 16-bit fixed point
            case 13: // 24-bit fixed point
            case 14: // 32-bit fixed point
            case 18: // 16-bit linear with emphasis
            case 19: // 16-bit linear compressed
            case 20: // 16-bit linear with emphasis and compression
            case 21: // Music kit DSP commands
            case 23: // ADPCM G.721
            case 24: // ADPCM G.722
            case 25: // ADPCM G.723.3
            case 26: // ADPCM G.723.5
            default:
              ppThrow (EUnsupported, "Unknown AU format");
        }
    }

      // NO header present
      // assume this is a 8khz mu-Law dump
      m_frequency = 8000;
      // read and decode data
      temp_data = new unsigned char [m_length];

      if (fread(temp_data, m_length, 1, fp)!=1)
        {
          ppThrow (EAccessFailure, "read error");
        }
      m_data0 = new unsigned char [m_length];

      Codec::DecodeMuLaw(temp_data, (unsigned short *&)m_data0, m_length);
      // multiply length by two (as samples are 16 bit)
      m_length <<=1;
      Set16Bit();
      return true;
    }
  catch (Exception &message)
    {
      ppWarning ("Exception caught: %s (%s) in %s",
                 message.Type (), message.Details (), message.Origin ());
      // we prboably want to free any allocated memory here
      return false;
    }
  catch (...)
    {
      // we prboably want to free any allocated memory here
      return false;
    }
}

//============================================================================
// SampleAu::save
//----------------------------------------------------------------------------
//
//============================================================================
bool SampleAu::Save(const char *filename, const Sample& sample) const
{
  AU_HEADER ah;
  FILE*     fp = fopen(filename, "wb");

  if (!fp)
    {
      perror("fopen");
      ppWarning ("error opening file");
      return false;
    }

  // fill in header details
  ah.header_size = Codec::BE32(28l);
  // ah.length will need to be changed if the data is
  // converted to u-law, A-law or ADPCM
  ah.length = Codec::BE32((unsigned long)sample.GetLength());
  // supported AU formats are
  // 1  :  8 bit mu-law
  // 27 :  8 bit A-law
  // 2  :  8 bit linear
  // 3  : 16 bit linear
  if (sample.Is16Bit())
    ah.format = Codec::BE32(3l);
  else
    ah.format = Codec::BE32(2l);

  ah.sample_rate = Codec::BE32((unsigned short)sample.GetFrequency());
  ah.channels    = Codec::BE32(1l);// only mono supported

  try
    {

      // leave space for filesize and come back later to write it
      if (fwrite(".snd    ", 8, 1, fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      if (fwrite(&ah, sizeof(AU_HEADER),1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }
      // write the optional text description (at least 4 bytes)
      if (fwrite("    ", 4,1,fp) != 1)
        {
          ppThrow (EAccessFailure, "write error");
        }

      switch (Codec::BE32(ah.format))
        {
        case 2:// 8 bit signed PCM
          {
            // convert to signed
            unsigned int i;
            for (i = 0; i < sample.GetLength(); ++i)
              sample.GetData()[i] ^= 0x80;

            if (fwrite(sample.GetData(), sample.GetLength(), 1, fp) != 1)
              {
                ppThrow (EAccessFailure, "write error");
              }

            // convert back to unsigned
            for (i = 0; i < sample.GetLength(); ++i)
              sample.GetData()[i] ^= 0x80;
          }
        case 3:// 16 bit PCM
          if (fwrite(sample.GetData(), sample.GetLength(), 1, fp) != 1)
            {
              ppThrow (EAccessFailure, "write error");
            }
        }
      return true;
    }
  catch (Exception &message)
    {
      ppWarning ("Exception caught: %s (%s) in %s",
                 message.Type (), message.Details (), message.Origin ());
      return false;
    }
  catch (...)
    {
      return false;
    }
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
