//=============================================================================
//
// directsound -- a class for playing sounds  -- windows DirectSound version
//
//-----------------------------------------------------------------------------
//
// $Id: DirectSound.cpp,v 1.1.1.1 1999/11/06 11:54:00 creinig Exp $
//
// $Log: DirectSound.cpp,v $
// Revision 1.1.1.1  1999/11/06 11:54:00  creinig
// Back in CVS at last
//
//
//=============================================================================
#include "PenguinPlay/DirectSound.h"

#include <math.h>

#include <string>

#ifndef DSBVOLUME_MAX
#define DSBVOLUME_MAX 0
#define DSBVOLUME_MIN -10000
#endif

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//===========================================================================
// CONSTRUCTORS & DESTRUCTORS
//---------------------------------------------------------------------------
//===========================================================================

///  Constructor
/// The rate set may not be the same as the rate given
/// bits - number of bits per sample
DirectSound::DirectSound(int rate, int bits, bool stereo)
   :  m_lpDS(0),
      m_timeout(0),
      m_rate(0),
      m_timer(0)
   {
   if (stereo)
      Create(rate,bits,1);
   else
      Create(rate,bits,0);

   // create the timeout event
   m_timeout_event   = CreateEvent(0, false, false, 0);
   TIMECAPS ptc;
   timeGetDevCaps(&ptc,sizeof(ptc));
   m_resolution = ptc.wPeriodMin;
   timeBeginPeriod(m_resolution);
   }

///  Destructor
DirectSound::~DirectSound()
   {
   try
     {
     Destroy();
     timeEndPeriod(m_resolution);
     }
   catch(...)
     {
     }
   }

//=====================================
// DirectSound COMMANDS
//=====================================

//===========================================================================
/// allocate voice
/// allocate channel
/// set voice to channel
/// start channel
/// wait till sample stops playing
/// deallocate channel
/// deallocate voice
//===========================================================================
Audio::Result DirectSound::Play(Sample& sample)
   {
   int v = AllocateVoice(sample);
   Channel* c = AllocateChannel();
   c->SetVoice(v);
   c->Start();
   Wait(1000*sample.GetSamples()/sample.GetFrequency());
   c->Stop();
   DeleteChannel(c);
   // deallocate the DirectSound buffer?

   return SUCCESS;
   }

//===========================================================================
void DirectSound::SetTimeout(long milliseconds)
   {
   m_timeout = milliseconds;
   }

void DirectSound::StartTimeout()
   {
   m_timer = timeSetEvent(m_timeout,
                          m_resolution,
                          (LPTIMECALLBACK)m_timeout_event,
                          0,
                          TIME_ONESHOT | TIME_CALLBACK_EVENT_SET);
   }

void DirectSound::StopTimeout()
   {
   if (m_timer != 0)
      timeKillEvent(m_timer);
   }


// if the time out hasn't occurred 2 seconds after the timeout should have
// occurred we can guess that it isn't going to happen
// perhaps it should be INFINITE
void DirectSound::WaitForTimeout()
   {
   if (m_timeout == 0)
      return;
   WaitForSingleObject(m_timeout_event, INFINITE);
   }



//===========================================================================
long DirectSound::GetPlaybackRate()
   {
   return m_rate;
   }

//===========================================================================
///
//===========================================================================
int DirectSound::AllocateVoice(const Sample& sample, int voice)
   {
   WAVEFORMATEX        waveformatex;
   DSBUFFERDESC        dsbd;
   LPDIRECTSOUNDBUFFER buffer;

   if (voice == -1)
      voice = VoicesAllocated()+1;

   // Create the secondary DirectSoundBuffer object to receive our sound data.
   // fill the waveformatex struct
   waveformatex.wFormatTag      = WAVE_FORMAT_PCM;
   waveformatex.nChannels       = sample.IsStereo() ? 2 : 1; // 1 or 2
   waveformatex.nSamplesPerSec  = sample.GetFrequency();
   waveformatex.wBitsPerSample  = sample.Is16Bit() ? 16 : 8; // 8 or 16
   waveformatex.nBlockAlign     = waveformatex.nChannels * waveformatex.wBitsPerSample >> 3;
   waveformatex.nAvgBytesPerSec = waveformatex.nSamplesPerSec * waveformatex.nBlockAlign;
   waveformatex.cbSize          = 0;

   //bufsize = waveformatex.nBlockAlign * 2024;

   memset( &dsbd, 0, sizeof( DSBUFFERDESC ));
   dsbd.dwSize        = sizeof( DSBUFFERDESC );
   //DSBCAPS_STICKYFOCUS - keep playing when the app has lost focus
   //                      unless new focus app is using directsound
   //DSBCAPS_GLOBALFOCUS - keep playing when the app has lost focus
   //                      unless another app with exclusive access starts
   //                      MSVC 97 doesn't seem to have this in the header
#ifndef DSBCAPS_GLOBALFOCUS
#define DSBCAPS_GLOBALFOCUS DSBCAPS_STICKYFOCUS
#endif
   dsbd.dwFlags       = DSBCAPS_CTRLALL | DSBCAPS_STATIC | DSBCAPS_GLOBALFOCUS ;
   dsbd.dwBufferBytes = sample.GetLength();
   dsbd.lpwfxFormat   = &waveformatex;



   HRESULT dsRetVal = m_lpDS->CreateSoundBuffer(&dsbd, &buffer, NULL);

   switch (dsRetVal)
      {
      case DS_OK:
         {
         // copy sample data to buffer.
         WriteSampleToBuffer(buffer, sample.GetData(),sample.GetLength());
         // insert the sound buffer in the vector
         m_voices[voice] = buffer;

         return voice;
         }

      case DSERR_ALLOCATED:     ppThrow (EMethodFailure, "DSERR_ALLOCATED");
      case DSERR_BADFORMAT:     ppThrow (EMethodFailure, "DSERR_BADFORMAT");
      case DSERR_INVALIDPARAM:  ppThrow (EInvalidParameter, "DSERR_INVALIDPARAM");
      case DSERR_NOAGGREGATION: ppThrow (EMethodFailure, "DSERR_NOAGGREGATION");
      case DSERR_OUTOFMEMORY:   ppThrow (EMethodFailure, "DSERR_OUTOFMEMORY");
      case DSERR_UNINITIALIZED: ppThrow (EMethodFailure, "DSERR_UNINITIALIZED");
      case DSERR_UNSUPPORTED:   ppThrow (EUnavailable, "DSERR_UNSUPPORTED");
      default:                  ppThrow (EMethodFailure, "DSERR_UNKNOWN");
      }
   }

//===========================================================================
///  CreateWritePrimaryBuffer
//===========================================================================
int DirectSound::CreateWritePrimaryBuffer(LPDIRECTSOUNDBUFFER* lplpDSB,
                                             int                  rate,
                                             int                  bits,
                                             int                  stereo)
   {
   WAVEFORMATEX waveformatex;
   DSBUFFERDESC dsbd;
   // Create the primary DirectSoundBuffer object to receive our sound data.

   memset(&waveformatex, 0, sizeof(WAVEFORMATEX));
   waveformatex.wFormatTag      = WAVE_FORMAT_PCM;
   waveformatex.nChannels       = stereo+1; // 1 or 2
   waveformatex.nSamplesPerSec  = rate;
   waveformatex.wBitsPerSample  = bits; // 8 or 16
   waveformatex.nBlockAlign     = waveformatex.nChannels * bits >> 3;
   waveformatex.nAvgBytesPerSec = rate * waveformatex.nBlockAlign;

   memset( &dsbd, 0, sizeof( DSBUFFERDESC ));
   dsbd.dwSize        = sizeof( DSBUFFERDESC );
   dsbd.dwFlags       = DSBCAPS_PRIMARYBUFFER;

   HRESULT dsRetVal = m_lpDS->CreateSoundBuffer(&dsbd, lplpDSB, NULL);

   if (dsRetVal != DS_OK)
      {
      return false;
      }

   dsRetVal = (*lplpDSB)->SetFormat(&waveformatex);
   if (dsRetVal != DS_OK)
      {
      return FALSE;
      }

   DWORD size;
   (*lplpDSB)->GetFormat(NULL, 0, &size);
   LPWAVEFORMATEX lpwfx = (LPWAVEFORMATEX)(new byte [size]);
   (*lplpDSB)->GetFormat(lpwfx, size, &size);

   if (lpwfx != 0)
      {
      m_rate = lpwfx->nSamplesPerSec;
      delete lpwfx;
      }

   // Get the primary buffer going
   (*lplpDSB)->Play(0,0,DSBPLAY_LOOPING);
   return TRUE;
   }

//===========================================================================

bool DirectSound::WriteSampleToBuffer(LPDIRECTSOUNDBUFFER  lpDsb,
                                         const unsigned char* lpbSoundData,
                                         DWORD                dwSoundBytes,
                                         DWORD                dwOffset)
   {
   LPVOID  lpvPtr1;
   DWORD   dwBytes1;
   LPVOID  lpvPtr2;
   DWORD   dwBytes2;
   HRESULT hr;
   // Obtain write pointer.
   hr = lpDsb->Lock(dwOffset,
                    dwSoundBytes,
                    &lpvPtr1,
                    &dwBytes1,
                    &lpvPtr2,
                    &dwBytes2,
                    0);
   // If DSERR_BUFFERLOST is returned, restore and retry lock.
   if (DSERR_BUFFERLOST == hr)
      {
      lpDsb->Restore();
      hr = lpDsb->Lock(dwOffset,
                       dwSoundBytes,
                       &lpvPtr1,
                       &dwBytes1,
                       &lpvPtr2,
                       &dwBytes2,
                       0);
      }
   if (DS_OK == hr)
      {
      // Write to pointers.
      CopyMemory(lpvPtr1, lpbSoundData, dwBytes1);
      if (NULL != lpvPtr2)
         {
         CopyMemory(lpvPtr2, lpbSoundData+dwBytes1, dwBytes2);
         }
      // Release the data back to DirectSound.
      hr = lpDsb->Unlock(lpvPtr1,
         dwBytes1,
         lpvPtr2,
         dwBytes2);
      if (DS_OK == hr)
         {            // Success.
         return TRUE;
         }
      }
   // Lock, Unlock, or Restore failed.
   return FALSE;
   }

//===========================================================================
///  Create the DirectSound object and set its cooperative level.
///  Create the DirectSoundBuffers.
//===========================================================================
void DirectSound::Create(int rate, int bits, int stereo)
   {
   // create the direct sound object
   if (DS_OK == DirectSoundCreate(NULL, &m_lpDS, NULL))
      {
      // Creation succeeded.
      if (DS_OK != m_lpDS->SetCooperativeLevel(GetForegroundWindow(),
         DSSCL_PRIORITY))
         {
         ppThrow (EMethodFailed, "Can't set cooperative level");
         }
      }
   else
      {
      ppThrow (EConstructionFailed, "Can't create DirectSound object");
      }

   CreateWritePrimaryBuffer(&m_lpDSBprimary, rate, bits, stereo);
   }

//===========================================================================
///  deallocate any dynamically allocated memory
//===========================================================================
void DirectSound::Destroy()
   {
   CloseHandle(m_timeout_event);
   StopTimeout();
   // release duplicate buffers
   ChannelContainer::iterator i   = m_channels.begin();
   ChannelContainer::iterator chan_end = m_channels.end();
   for (;i != chan_end; ++i)
      {
      ChannelVoice& v = i->second;
      if (v.m_duplicate && v.m_buffer != 0)
         v.m_buffer->Release();
      v.m_buffer = 0;
      }
   // release all directsound buffers
   VoiceContainer::iterator j   = m_voices.begin();
   VoiceContainer::iterator end = m_voices.end();
   for (;j != end; ++j)
      j->second->Release();
   m_voices.clear();
   m_lpDS->Release();
   }

//===========================================================================
///  check the status of the directsound object
///  \return true if it is playing
//===========================================================================
bool DirectSound::IsPlaying(LPDIRECTSOUNDBUFFER buffer)
   {
   DWORD dwStatus;
   buffer->GetStatus(&dwStatus);
   return (dwStatus & DSBSTATUS_PLAYING) != 0;
   }

//===========================================================================
///  \return the number of voices (ie direct sound buffers)
/// that have been created
int DirectSound::VoicesAllocated() const
   {
   return m_voices.size();
   }

//===========================================================================
///  allocate a new channel, and put it in the container
Channel* DirectSound::AllocateChannel()
   {
   Channel* ret = new Channel(*this, m_channels.size());
   m_channels[m_channels.size()].m_id        = -1; // -1 -> no voice assigned
   m_channels[m_channels.size()].m_duplicate = false;
   m_channels[m_channels.size()].m_buffer    = 0;
   return ret;
   }

//===========================================================================
///   delete the given channel
void DirectSound::DeleteChannel(Channel*& channel)
   {
   m_channels.erase(channel->GetID());
   delete channel;
   channel = 0;
   }

//===========================================================================

Audio::Result DirectSound::SetLocation(int x, int y, int z, int channel)
   {
   Result result;
   result = SetX(x, channel);
   if (result != SUCCESS)
      return result;
   result = SetY(y, channel);
   if (result != SUCCESS)
      return result;
   result = SetZ(z, channel);
   return result;
   }

//===========================================================================

Audio::Result DirectSound::SetX(int x, int channel)
   {
   return NOT_AVAILABLE;
   }

//===========================================================================

Audio::Result DirectSound::SetY(int y, int channel)
   {
   return NOT_AVAILABLE;
   }

//===========================================================================

Audio::Result DirectSound::SetZ(int z, int channel)
   {
   return NOT_AVAILABLE;
   }

//===========================================================================

int DirectSound::GetX(int channel)
   {
   return -1;
   }

//===========================================================================

int DirectSound::GetY(int channel)
   {
   return -1;
   }

//===========================================================================

int DirectSound::GetZ(int channel)
   {
   return -1;
   }

// DirectSound volume is logarithmic
static inline long ConvertToDirectSoundVolume(Audio::Volume volume)
   {
   return ((log(volume) * (DSBVOLUME_MAX - DSBVOLUME_MIN))
            / log(Audio::MAX_VOLUME)+DSBVOLUME_MIN);
   }

static inline Audio::Volume ConvertFromDirectSoundVolume(long volume)
   {
   return pow(((volume - DSBVOLUME_MIN) * Audio::MAX_VOLUME)
            / (DSBVOLUME_MAX - DSBVOLUME_MIN),10.0);
   }


//===========================================================================
// need to set the volume between DSBVOLUME_MAX (no attenuation) and DSBVOLUME_MIN
// 0 (MAX) .. -10,000 (MIN)
Audio::Result DirectSound::SetVolume(Audio::Volume volume,
                                           int              channel)
   {
   int voice = GetVoice(channel);
   if (voice == -1)
      return FAILURE;

   LPDIRECTSOUNDBUFFER lpdsb = m_voices[voice];
   if (lpdsb == 0)
      return FAILURE;
   lpdsb->SetVolume(ConvertToDirectSoundVolume(volume));
   return SUCCESS;
   }

//===========================================================================

DirectSound::Volume DirectSound::GetVolume(int channel)
   {
   int voice = GetVoice(channel);
   if (voice == -1)
      return 0;

   LPDIRECTSOUNDBUFFER lpdsb = m_voices[voice];
   if (lpdsb == 0)
      return 0;

   long volume;
   lpdsb->GetVolume(&volume);

   return ConvertFromDirectSoundVolume(volume);
   }

//===========================================================================

Audio::Result DirectSound::SetVelocity(Velocity velocity,
                                             int      channel)
   {
   return NOT_AVAILABLE;
   }

//===========================================================================

Audio::Volume DirectSound::GetVelocity(int channel)
   {
   return -1;
   }

//===========================================================================
// Set frequency between DSBFREQUENCY_MIN to DSBFREQUENCY_MAX
// need to take the format of the buffer in to account
// DSBFREQUENCY_ORIGINAL will set the frequency to the default
Audio::Result DirectSound::SetFrequency(Frequency frequency,
                                              int       channel)
   {
   int voice = GetVoice(channel);
   if (voice == -1)
      return FAILURE;

   LPDIRECTSOUNDBUFFER lpdsb = m_voices[voice];
   if (lpdsb == 0)
      return FAILURE;

   lpdsb->SetFrequency(frequency);

   return SUCCESS;
   }

//===========================================================================
Audio::Frequency DirectSound::GetFrequency(int channel)
   {
   int voice = GetVoice(channel);
   if (voice == -1)
      return 0;

   LPDIRECTSOUNDBUFFER lpdsb = m_voices[voice];
   if (lpdsb == 0)
      return 0;

   unsigned long frequency;
   lpdsb->GetFrequency(&frequency);

   return frequency;
   }


//===========================================================================

Audio::Result DirectSound::SetPosition(unsigned long position,
                                             int           channel)
   {
   LPDIRECTSOUNDBUFFER voice = m_channels[channel].m_buffer;
   if (voice == 0)
      return FAILURE;

   if (voice->SetCurrentPosition((DWORD)position) == DS_OK)
      return SUCCESS;
   else
      {
      return FAILURE;
      }
   }

//===========================================================================

unsigned long DirectSound::GetPosition(int channel)
   {

   LPDIRECTSOUNDBUFFER voice = m_channels[channel].m_buffer;

   if (voice == 0)
      return 0;

   DWORD play_cursor;
   DWORD write_cursor;

   if (voice->GetCurrentPosition(&play_cursor,
                                 &write_cursor) != DS_OK)
      {
      ppThrow (EAccessFailure, "Can't Get Current Position");
      }

   return play_cursor;
   }

//===========================================================================

Audio::Result DirectSound::SetVoice(int voice, int channel)
   {
   m_channels[channel].m_id = voice;
   Start(channel);
   return SUCCESS;
   }

//===========================================================================

int DirectSound::GetVoice(int channel)
   {
   return m_channels[channel].m_id;
   }

//===========================================================================

void DirectSound::Stop(int channel)
   {
   ChannelVoice& voice = m_channels[channel];

   if (voice.m_buffer != 0)
      voice.m_buffer->Stop();

   if (voice.m_buffer != 0 && voice.m_duplicate)
      {
      voice.m_buffer->Release();
      voice.m_buffer = 0;
      voice.m_duplicate = false;
      }

   }

//===========================================================================
/// This is not yet finished, looping samples are not handled correctly yet.
/// may need to create a duplicate so that samples can be played more
/// than once at any instant.
void DirectSound::Start(int channel)
   {
   Stop(channel);
   ChannelVoice& voice = m_channels[channel];
   if (voice.m_id != -1)
      {
      if (m_voices[voice.m_id] != 0)
         {
         if (IsPlaying(m_voices[voice.m_id]))
            {
            voice.m_duplicate = true;
            m_lpDS->DuplicateSoundBuffer(m_voices[voice.m_id],
                                         &voice.m_buffer);

            }
         else
            {
            voice.m_buffer = m_voices[voice.m_id];
            voice.m_duplicate      = false;
            }
         voice.m_buffer->Play(0,0,0);
         }
      }

   }

//===========================================================================

int DirectSound::NextChannel()
   {
   return m_channels.size();
   }

PP_I_NAMESPACE_END
PP_NAMESPACE_END
