//=============================================================================
// $Id: AudioToFile.cpp,v 1.3 1999/11/28 23:47:25 creinig Exp $
//-----------------------------------------------------------------------------
// $Log: AudioToFile.cpp,v $
// Revision 1.3  1999/11/28 23:47:25  creinig
// adapted sources to use ppconfig(_win32).h
//
// Revision 1.2  1999/11/20 22:05:46  pcburns
// Added cast to stop signed/unsigned warnings.
// Added #ifdefs for NO_MIXER_THREAD.
// Added code to WaitForTimeout for NO_MIXER_THREAD so that mixing
// occurs for the complete time.
//
// Revision 1.1.1.1  1999/11/06 11:54:00  creinig
// Back in CVS at last
//
//=============================================================================

#include "PenguinPlay/AudioToFile.h"
#include "PenguinPlay/Sample.h"
#include "PenguinPlay/Channel.h"
#include "PenguinPlay/SoftwareMixer.h"

#include <string>
#ifdef PP_HAVE_UNISTD_H
#include <unistd.h> // sleep, close, access
#else
#include <io.h>
#define close _close
#endif
#include <stdio.h>
#include <signal.h>
#include <memory.h>
#include <assert.h>
#include <sys/types.h> // u_int8_t, int16_t
#ifdef PP_HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#ifndef NO_MIXER_THREAD
#define NO_MIXER_THREAD
#endif
#endif
PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

const int   AudioToFile::DEFAULT_BUFFER_SIZE = 32768;

#ifndef NO_MIXER_THREAD
static itimerval* sg_old_it = 0;
#endif

//=============================================================================

AudioToFile::AudioToFile()
  :
#ifndef NO_MIXER_THREAD
     m_mixer_thread(this),
#endif
     m_audio_buffer(0),
     m_channels_enabled(0),
     m_fd(-1),
     m_mixer_started(false)
{

}

//=============================================================================

AudioToFile::~AudioToFile()
{
     try
          {
               if (m_fd != -1)
                    {
                         close(m_fd);
                         m_fd = -1;
                    }
               delete [] m_audio_buffer;
               m_audio_buffer = 0;
          }
     catch(...)
          {
          }
}



//=============================================================================
/// \return The position converted from fixed point.
//=============================================================================
inline Fixed AudioToFile::Status::GetPosition() const
{
     return m_position;
}

//=============================================================================
/// Set the position -- convert to fixed point.
//=============================================================================
inline void AudioToFile::Status::SetPosition(Fixed position)
{
     m_position = position;
}

//=============================================================================

int AudioToFile::AllocateVoice(const Sample& sample, int voice)
{
     if (voice == -1)
          voice = VoicesAllocated()+1;
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_voices[voice] = &sample;
     return voice;
}

//=============================================================================

Channel* AudioToFile::AllocateChannel()
{
     Channel* ret = new Channel(*this);
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[m_channels.size()].m_voice = -1;
     return ret;
}

//=============================================================================

void AudioToFile::DeleteChannel(Channel*& channel)
{
     delete channel;
     channel = 0;
}

//=============================================================================

int AudioToFile::VoicesAllocated() const
{
     return m_voices.size();
}

//=============================================================================

AudioToFile::Result AudioToFile::SetLocation(int x,
                                             int y,
                                             int z,
                                             int channel)
{
     Result result = SetX(x, channel);
     if (result != SUCCESS)
          return result;
     result = SetY(y, channel);
     if (result != SUCCESS)
          return result;
     result = SetZ(z, channel);
     return result;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetX(int x, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_x = x;
     return SUCCESS;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetY(int y, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_y = y;
     return SUCCESS;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetZ(int z, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_z = z;
     return SUCCESS;
}

//=============================================================================

int AudioToFile::GetX(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_x;
}

//=============================================================================

int AudioToFile::GetY(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_y;
}

//=============================================================================

int AudioToFile::GetZ(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_z;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetVolume(Volume volume, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_volume = volume;
     return SUCCESS;
}

//=============================================================================

AudioToFile::Volume AudioToFile::GetVolume(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_volume;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetVelocity(Velocity velocity, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_velocity = velocity;
     return SUCCESS;
}

//=============================================================================

AudioToFile::Velocity AudioToFile::GetVelocity(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_velocity;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetFrequency(Frequency frequency,
                                              int       channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_frequency = frequency;
     return SUCCESS;
}

//=============================================================================

AudioToFile::Frequency AudioToFile::GetFrequency(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_frequency;
}

//=============================================================================

AudioToFile::Result AudioToFile::SetPosition(unsigned long position,
                                             int           channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     const Sample* s = m_channels[channel].m_sample;
     if (s && position < s->GetSamples())
          {
               m_channels[channel].SetPosition(Fixed(position));
               return SUCCESS;
          }
     else
          return FAILURE;
}

//=============================================================================

unsigned long AudioToFile::GetPosition(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].GetPosition().c_long();
}

//=============================================================================

AudioToFile::Result AudioToFile::SetVoice(int voice, int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     m_channels[channel].m_voice = voice;
     m_channels[channel].m_sample = m_voices[voice];
     // start ?
     return SUCCESS;
}

//=============================================================================

int AudioToFile::GetVoice(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels[channel].m_voice;
}

//=============================================================================

int AudioToFile::NextChannel()
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels.size()+1;
}

//=============================================================================
/// Start the playback sample mixing thread if it hasn't already been
/// started and enable the channel for mixing.
///
/// We have two options for the mixer. Either fork a new process and
/// communicate via a pipe or perhaps shared memory, or start a new thread for
/// the mixer.
/// Some libraries are not thread safe so it may be a good idea to
/// fork a process for the mixer.
//=============================================================================
void AudioToFile::Start(int channel)
{
     if (!m_channels[channel].m_started)
          {
               m_channels[channel].m_started = true;
               ++m_channels_enabled;
          }

     if (!m_mixer_started)
          {
               // you might want not want to create a thread when debugging
               // the mixing thread doesn't stop though

#if NO_MIXER_THREAD
               //      MixerThread();
#else
          m_mixer_thread.Start();
          m_mixer_started = true;
#endif
          }
}

//=============================================================================
///
//=============================================================================
void AudioToFile::Stop(int channel)
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     if (m_channels[channel].m_started)
          {
               m_channels[channel].m_started = false;
               --m_channels_enabled;
          }
}

//=============================================================================
/// play the sample, wait till it finishes and then return
//=============================================================================
Audio::Result AudioToFile::Play(Sample& sample)
{
     int v = AllocateVoice(sample);
     Channel* c = AllocateChannel();
     c->SetVoice(v);
     c->SetFrequency(sample.GetFrequency());
     c->Start();
     // wait till sample stops playing?
     // time for one loop = number of samples / samples per second

     SetTimeout(1000*sample.GetSamples()/sample.GetFrequency());
     StartTimeout();
     WaitForTimeout();
     c->Stop();
     // deallocate channel
     DeleteChannel(c);
     // deallocate voice ?

     return SUCCESS;
}

//=============================================================================
///
//=============================================================================
inline Fixed AudioToFile::Status::GetContiguousLength(Fixed scale)
{
     Fixed length;
     if (m_reversing)
          {
               length = m_position.c_long() - m_sample->GetLoopStart();
          }
     else
          {
               length = m_sample->GetLoopEnd() - m_position.c_long();
          }

     // make scale absolute so we can scale the length
     if (scale < Fixed(0.0))
          scale = -scale;

     if (scale >= length)
          return Fixed(1.0);

     return length / scale;
}

//=============================================================================
///
//=============================================================================
inline void AudioToFile::Status::EndOfBidiLoop(Fixed scale)
{
     m_reversing = !m_reversing;

     if (m_reversing)
          {
               m_position = Fixed(m_sample->GetLoopEnd()-1<<1) - m_position;
               assert((unsigned long)GetPosition().c_long()
                      < m_sample->GetLoopEnd());
          }
     else if ((Sample::position_t)m_position.c_long() > m_sample->GetLoopEnd())
          {
               m_position = Fixed(m_sample->GetLoopStart()<<1)-m_position;
               assert((unsigned long)GetPosition().c_long()
                      < m_sample->GetLoopEnd());
          }
     else if ((unsigned long)m_position.c_long()
              < m_sample->GetLoopStart())
          {
               m_position = Fixed(m_sample->GetLoopStart()<<1)-m_position;
               assert((unsigned long)GetPosition().c_long()
                      < m_sample->GetLoopEnd());
          }
}

//=============================================================================
///
//=============================================================================
inline void AudioToFile::Status::EndOfLoop(Fixed scale)
{
#if 0
     //      m_position -= m_sample->GetLoopStart()<<FIXEDBITS;

     m_position  = m_position % (m_sample->GetLoopLength());
     m_position += Fixed(m_sample->GetLoopStart());

     assert((Sample::position_t)GetPosition().c_long() <
            m_sample->GetLoopEnd());
#else
     if ((Sample::position_t)scale.c_long() > m_sample->GetLoopLength())
          {
               m_position
                    = Fixed(m_position.c_long()
                               % (m_sample->GetLoopLength()+1))
                    + m_position.fractional();
               assert(GetPosition().c_long() < (long)m_sample->GetLoopEnd());
          }
     else if ((Sample::position_t)GetPosition().c_long() >
              m_sample->GetLoopLength())
          {
               m_position -= Fixed(m_sample->GetLoopLength());
               assert(GetPosition().c_long() < (long)m_sample->GetLoopEnd());
          }
     else
          {
               m_position = Fixed(m_sample->GetLoopStart());
               assert(GetPosition().c_long() < (long)m_sample->GetLoopEnd());
          }
#endif

}

//=============================================================================
///
//=============================================================================
inline void AudioToFile::Status::EndOfContiguousLength(Fixed scale)
{
     if (m_sample->IsBIDI())
          {
               EndOfBidiLoop(scale);
          }
     else
          {
               EndOfLoop(scale);
          }
}

//=============================================================================
/// \param status
/// \param length
/// \param scale
//=============================================================================
inline void AudioToFile::MixBuffer(Status&       status,
                                   unsigned long length,
                                   Fixed         scale)
{
     unsigned long left;
     unsigned long right;

     status.CalculateVolume(left, right);


     // what about the mono version ?
     // at the moment it always mix to stereo and the mono output
     // functions use only one channel of the stereo output

     if (status.m_sample->Is16Bit())
          {
          SoftwareMixer::MixStereoUnrolled<int16_t, int32_t, 10>
                    (length,
                     left,
                     right,
                     scale,
                     status.m_position,
                     (int16_t*)status.m_sample->GetData(),
                     (int32_t*)m_audio_buffer);
          }
     else
          {
          SoftwareMixer::MixStereoUnrolled<u_int8_t, int32_t, 10>
                    (length,
                     left,
                     right,
                     scale,
                     status.m_position,
                     (u_int8_t*)status.m_sample->GetData(),
                     (int32_t*)m_audio_buffer);
          }
}


//=============================================================================
/// \param status
/// \param scale
//=============================================================================
inline void AudioToFile::MixLoopedChannel(Status& status, Fixed scale)
{
     unsigned long playlength = m_mix_length;

     if (status.m_reversing)
          scale = -scale;

     // mix the sample into the mix_buffer
     while (playlength != 0)
          {
               bool shortened = false;
               unsigned long length = status.GetContiguousLength(scale).c_long();
               if (length > playlength)
                    {
                         length     = playlength;
                         playlength = 0;
                         shortened  = true;
                    }
               else
                    {
                         playlength -= length;
                    }
               MixBuffer(status, length, scale);
               // at end of loop or end of playlength
               if (!shortened)
                    {
                         status.EndOfContiguousLength(scale);
                    }
          }
}

//=============================================================================
/// \param status
/// \param scale
//=============================================================================
inline void AudioToFile::MixOneShotChannel(Status& status, Fixed scale)
{
     if (status.m_sample->GetLength() <
         (Sample::position_t)status.GetPosition().c_long())
          {
               // sample has finished -- don't play it
               status.m_frequency = 0;
               status.m_position  = 0;
               return;
          }

     // playlength of sample left to play
     Fixed playlength = (Fixed(status.m_sample->GetLength())
                            - status.m_position);
     playlength /= scale;
     if ((unsigned long)playlength.c_long() > m_mix_length)
          playlength = m_mix_length;

     // mix the sample into the mix_buffer
     MixBuffer(status, playlength.c_long(), scale);
}

//=============================================================================
///
//=============================================================================
void AudioToFile::MixChannel(Status& status)
{
     if (!status.m_started            ||
         !status.m_sample             ||
         !status.m_sample->GetData0() ||
         !status.m_frequency)
          return;

     // calculate the scale
     // we may be able to speed this up by only calculating the scale
     // when the frequency changes and putting the result into status
     // using a Fixed instead of a double causes overflow since
     // rate is usually larger than status.m_frequency
     Fixed scale = Fixed((double)status.m_frequency / m_rate);

     // calculate the length of the sample to mix
     if (status.m_sample->IsLooped())
          {
               MixLoopedChannel(status, scale);
          }
     else
          {
               MixOneShotChannel(status, scale);
          }
}

//=============================================================================
///  Convert the audio buffer from 32 bit stereo to 16 bit stereo.
//=============================================================================
// unrolled loop
inline void AudioToFile::Output32_16stereo()
{
     // is this sort of aliasing really bad for performance?
     int32_t*  in  = (int32_t  *)m_audio_buffer;
     u_int8_t* out = (u_int8_t *)m_audio_buffer;

     size_t j         = m_mix_length >> 2;
     size_t remainder = m_mix_length & 3; // mod 4

     while (j--)
          {
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;

               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;

               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;

               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
          }

    while (remainder--)
          {
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
               *out++ =  (*in   >> 16)  & 0xff;
               *out++ =  (*in++ >> 24)  & 0xff;
          }

     if (write(m_fd, m_audio_buffer, m_mix_length<<2) == -1)
        ppThrow (EAccessFailure, "Error writing to /dev/dsp");
}


//=============================================================================
/// Convert the audio buffer from 32 stereo to 16 bit mono.
//=============================================================================
inline void AudioToFile::Output32_16mono()
{
     int32_t*  in  = (int32_t  *)m_audio_buffer;
     u_int8_t* out = (u_int8_t *)m_audio_buffer;

     size_t j         = m_mix_length >> 2;
     size_t remainder = m_mix_length & 3; // mod 4

     while (j--)
          {
               // copy to buffer -- low byte then high byte
               *out++ = *in >> 16 & 0xff;
               *out++ = *in >> 24 & 0xff;
               in    += 2;
               *out++ = *in >> 16 & 0xff;
               *out++ = *in >> 24 & 0xff;
               in    += 2;
               *out++ = *in >> 16 & 0xff;
               *out++ = *in >> 24 & 0xff;
               in    += 2;
               *out++ = *in >> 16 & 0xff;
               *out++ = *in >> 24 & 0xff;
               in    += 2;
          }
     while (remainder--)
          {
               *out++ = *in   >> 16 & 0xff;
               *out++ = *in++ >> 24 & 0xff;
          }
     if (write(m_fd, m_audio_buffer, m_mix_length<<1) == -1)
          ppThrow (EAccessFailure, "Error writing to /dev/dsp");
}

//=============================================================================
// loop unrolled
inline void AudioToFile::Output32_8stereo()
{
     // convert to 8 bit unsigned and copy to buffer
     // -- buffer should be big enough for player
     int32_t*  in  = (int32_t  *)m_audio_buffer;
     u_int8_t* out = (u_int8_t *)m_audio_buffer;

     size_t j         = m_mix_length >> 2;
     size_t remainder = m_mix_length & 3; // mod 4
     while (j--)
          {
               // copy to buffer and convert to unsigned
               // left first then right
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
          }
     while (remainder--)
          {
               *out++ = (*in++ >> 24) ^ 0x80;
               *out++ = (*in++ >> 24) ^ 0x80;
          }
     // write is in bytes multiply by two
     if (write(m_fd, m_audio_buffer, m_mix_length<<1) == -1)
          ppThrow (EAccessFailure, "Error writing to /dev/dsp");
}


//=============================================================================
inline void AudioToFile::Output32_8mono()
{
     // convert to 8 bit unsigned and copy to buffer
     // -- buffer should be big enough for player
     int32_t*  in  = (int32_t  *)m_audio_buffer;
     u_int8_t* out = (u_int8_t *)m_audio_buffer;

     size_t j         = m_mix_length >> 2;
     size_t remainder = m_mix_length & 3; // mod 4

     while (j--)
          {
               // copy to buffer and convert to unsigned
               *out++ = (*in >> 24) ^ 0x80; in += 2;
               *out++ = (*in >> 24) ^ 0x80; in += 2;
               *out++ = (*in >> 24) ^ 0x80; in += 2;
               *out++ = (*in >> 24) ^ 0x80; in += 2;
          }

     while (remainder--)
          {
               *out++ = (*in >> 24) ^ 0x80;
               in+=2;

          }
     if (write(m_fd, m_audio_buffer, m_mix_length) == -1)
          ppThrow (EAccessFailure, "Error writing to /dev/dsp");
}

//=============================================================================

void AudioToFile::Output()
{
     if (m_bits == 16)
          {
               if (m_stereo) Output32_16stereo();
               else          Output32_16mono();
          }
     else
          {
               if (m_stereo) Output32_8stereo();
               else          Output32_8mono();
          }
}

inline void AudioToFile::PrepareBuffer()
{
     // clear the buffer
     memset(m_audio_buffer, 0, m_buffer_size);
}


//=============================================================================

inline void AudioToFile::Mix()
{
     PrepareBuffer();

     // enumerate the channels
     ChannelContainer::iterator i   = m_channels.begin();
     ChannelContainer::iterator end = m_channels.end();
     for (;i != end;++i)
          MixChannel(i->second);
     // write buffer to /dev/dsp
     Output();
}


//=============================================================================

inline bool AudioToFile::KeepMixing()
{
#ifndef NO_MIXER_THREAD
     MutexLock lock(m_mutex);
#endif
     return m_channels_enabled > 0;
}



//=============================================================================
///  This thread mixes all the active channels together and writes the result
///  to /dev/dsp
//=============================================================================
void AudioToFile::MixerThread()
{
     // make sure that the mix buffer is allocated
     if (m_audio_buffer == 0)
          {
               ppThrow (ENullPointer, "No mix buffer allocated!");
          }

     try
          {
               while (KeepMixing())
                    {
                         Mix();
                    }

               // clear the buffer when it has finished playing
               PrepareBuffer();
          }
     catch(...)
          {
               ppWarning ("Exception caught in AudioToFile::MixerThread()");
          }
     m_mixer_started = false;
}

//=============================================================================
///  This function is given to pthread_create as the callback.
///  It must be static, otherwise the hidden "this" pointer will get corrupted.
///  The argument given in me should be the "this" pointer so that we can
///  get back inside the class again.
//=============================================================================
void* AudioToFile::thread_callback(void* me)
{
     AudioToFile* dsp = (AudioToFile*)me;
     dsp->MixerThread();
     return NULL;
}


//=============================================================================

long AudioToFile::GetPlaybackRate()
{
     return m_rate;
}

//=============================================================================

void AudioToFile::SetTimeout(long milliseconds)
{
     m_timeout = milliseconds;
     // need to also change the mix length so we can be responsive
     // to these time outs
     unsigned long length = (m_rate * m_timeout) / 1000;
#if 0
     int shift = 4; // buffer needs to accomodate 32 bit size samples for mixer
     if (m_stereo)
          shift<<=1;
#else
     // 32 bits (4 bytes) plus always stereo
     int shift = 8;
#endif
     //     if (m_bits == 16)
     //          shift<<=1;
     m_mix_length = (((length*shift) < m_buffer_size)
                     ? length : (m_buffer_size/shift));
}

//=============================================================================

void AudioToFile::StartTimeout()
{
#ifndef NO_MIXER_THREAD
     // m_timeout is in milliseconds
     // divide by 1000 to get seconds
     // modulus 1000 is microseconds

     // block the signal so that we wont
     // get it until we call WaitForTimeout

     // mask SIGALRM
#if 0
     sigset_t sigset;
     sigemptyset(&sigset);
     sigaddset(&sigset, SIGALRM);
     pthread_sigmask(SIG_BLOCK, &sigset, 0);
#endif
     ::itimerval it;
     it.it_interval.tv_sec  = 0;
     it.it_interval.tv_usec = 0;
     it.it_value.tv_sec     = m_timeout/1000;
     it.it_value.tv_usec    = 1000*(m_timeout%1000);
     //     setitimer(ITIMER_REAL, &it, sg_old_it);
     setitimer(ITIMER_REAL, &it, 0);
#endif
}

//=============================================================================

void AudioToFile::StopTimeout()
{

}

//=============================================================================

void AudioToFile::WaitForTimeout()
{
#ifndef NO_MIXER_THREAD
     int      sig = 0;
     sigset_t sigset;

     sigemptyset(&sigset);
     sigaddset(&sigset, SIGALRM);
     sigwait(&sigset, &sig);

     setitimer(ITIMER_REAL, 0, 0);
     //     setitimer(ITIMER_REAL, sg_old_it, sg_old_it);
#else
     int times = (m_rate * m_timeout * 8) / (1000 * m_mix_length);
     if (times == 0) times = 1;
     while (times--)
          Mix();
#endif
}


PP_I_NAMESPACE_END
PP_NAMESPACE_END
