#include <PenguinPlay/PenguinPlay.h>
#include <PenguinPlay/AudioCD.h>

#ifdef PP_SYS_OS_LINUX

// <linux/cdrom.h> expects u_char and caddr_t to be defined
// they may be <sys/types.h>

#include <sys/types.h>
#ifndef u_char
#define u_char unsigned char
#endif
#ifndef caddr_t
#define caddr_t char*
#endif

#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//============================================================================
class AudioCD::AudioCDPrivate
{
public:
  AudioCDPrivate()
    : m_device("/dev/cdrom")
    {
    m_cd = open(m_device, O_RDONLY | O_NONBLOCK);
    if (m_cd == -1)
      ppThrow (EAccessFailure, "Error opening cd for reading");
    }

  ~AudioCDPrivate()
    {
    close(m_cd);
    m_cd = -1;
    }

  int   m_cd;
  char* m_device;
};

//============================================================================

void AudioCD::Stop()
{
  ioctl(m_private->m_cd, CDROMSTOP);
}

//============================================================================

void AudioCD::Pause()
{
  ioctl(m_private->m_cd, CDROMPAUSE);
}

//============================================================================

void AudioCD::Resume()
{
  ioctl(m_private->m_cd, CDROMRESUME);
}

//============================================================================

void AudioCD::Eject()
{
  ioctl(m_private->m_cd, CDROMEJECT);
}

//============================================================================

void AudioCD::Retract()
{
  ioctl(m_private->m_cd, CDROMCLOSETRAY);
}

//============================================================================

void AudioCD::Play(int start, int end)
{
  cdrom_ti trackind;
  memset(&trackind, 0, sizeof(cdrom_ti));
  trackind.cdti_trk0 = start;
  trackind.cdti_trk1 = end;
  Stop();
  ioctl(m_private->m_cd, CDROMPLAYTRKIND, &trackind);
}

//============================================================================

void AudioCD::Play(int track)
{
  Play(track, track);
}

//============================================================================

void AudioCD::Play()
{
  Play(GetFirstTrack(), GetLastTrack());
}

//============================================================================

void AudioCD::GetFirstAndLast(int& first, int& last)
{
  cdrom_tochdr tochdr;
  ioctl(m_private->m_cd, CDROMREADTOCHDR, &tochdr);
  first = tochdr.cdth_trk0;
  last  = tochdr.cdth_trk1;
}

//============================================================================

int AudioCD::GetFirstTrack()
{
  int first, last;
  GetFirstAndLast(first,last);
  return first;
}

//============================================================================

int AudioCD::GetLastTrack()
{
  int first, last;
  GetFirstAndLast(first,last);
  return last;
}

//============================================================================

int AudioCD::GetCurrentTrack()
{
  cdrom_subchnl subchnl;
  subchnl.cdsc_format = CDROM_MSF;
  ioctl(m_private->m_cd, CDROMSUBCHNL, &subchnl);
  return subchnl.cdsc_trk;
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END


#endif /* defined(PP_SYS_OS_LINUX) */

#ifdef PP_SYS_OS_WIN32
#include <windows.h>
#include <mmsystem.h>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//============================================================================
class AudioCD::AudioCDPrivate
{
public:
  AudioCDPrivate();
  ~AudioCDPrivate();

  static AudioCDPrivate* Create();

  bool Open();
  void Close();

  enum { MAX_TRACKS = 100 };

  MCI_OPEN_PARMS   m_mci_open;
  MCI_STATUS_PARMS m_mci_status;
  short            m_tracks;
  short            m_track_length[MAX_TRACKS];
};

//============================================================================
// try to create a AudioCDPrivate
// without having to worry about exceptions
AudioCD::AudioCDPrivate* AudioCD::AudioCDPrivate::Create()
{
  AudioCDPrivate* p = 0;
  try
    {
     p = new AudioCDPrivate;
    }
  catch(...)
    {
    return 0;
    }
  return p;
}

//============================================================================

AudioCD::AudioCDPrivate::AudioCDPrivate()
  : m_tracks(0)
  {

  // open the cd

  if (!Open())
    {
    ppThrow (EAccessFailure, "Couldn't open CD player");
    }

  // Get the number of tracks on the cd

  m_mci_status.dwItem = MCI_STATUS_NUMBER_OF_TRACKS;
  if (mciSendCommand(m_mci_open.wDeviceID,
                     MCI_STATUS,
                     MCI_STATUS_ITEM | MCI_WAIT,
                     (DWORD)(LPVOID)&m_mci_status))
    {
    Close();
    ppThrow (EUnavailable, "Error getting number of tracks");
    }
  m_tracks = (short)m_mci_status.dwReturn;
  if (m_tracks > MAX_TRACKS)
      m_tracks = MAX_TRACKS;

  // Get the length of each track

  for (int i = 0; i < m_tracks; ++i)
    {
    m_mci_status.dwTrack = i+1;
    mciSendCommand(m_mci_open.wDeviceID,
                   MCI_STATUS,
                   MCI_TRACK | MCI_STATUS_ITEM | MCI_WAIT,
                   (DWORD)(LPVOID)&m_mci_status);
    m_track_length[i] = (short)(MCI_MSF_MINUTE(m_mci_status.dwReturn)*60+
                                MCI_MSF_SECOND(m_mci_status.dwReturn));

    }
  Close();
  }

//============================================================================

AudioCD::AudioCDPrivate::~AudioCDPrivate()
  {
    try
      {
        Close();
      }
    catch (...)
      {
      }
  }

//============================================================================

bool AudioCD::AudioCDPrivate::Open()
  {
  m_mci_open.lpstrDeviceType = (LPCTSTR)MCI_DEVTYPE_CD_AUDIO;
  int error = mciSendCommand(0,
                             MCI_OPEN,
                             MCI_OPEN_TYPE | MCI_OPEN_TYPE_ID | MCI_OPEN_SHAREABLE,
                             (DWORD)(LPVOID)&m_mci_open);
  if (error != 0)
    {
    char strError[128];
    mciGetErrorString(error, strError, 128);
    }
  return error == 0;

  }

//============================================================================

void AudioCD::AudioCDPrivate::Close()
  {
  mciSendCommand(m_mci_open.wDeviceID, MCI_CLOSE, 0, 0);
  }

//============================================================================

void AudioCD::Play(int start, int end)
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }

  MCI_SET_PARMS  mci_set;
  MCI_PLAY_PARMS mci_play;

  if (!m_private->Open())
    {
    ppThrow (EAccessFailure, "Couldn't open CD player");
    }

  mci_set.dwTimeFormat = MCI_FORMAT_TMSF;
  if (mciSendCommand(m_private->m_mci_open.wDeviceID,
                     MCI_SET,
                     MCI_SET_TIME_FORMAT,
                     (DWORD)&mci_set))
    {
    m_private->Close();
    return;
    }

  mci_play.dwCallback = 0;
  mci_play.dwFrom = MCI_MAKE_TMSF(start, 0, 0, 0);
  mci_play.dwTo = MCI_MAKE_TMSF(end, 0, 0, 0);
  if (mciSendCommand(m_private->m_mci_open.wDeviceID,
                     MCI_PLAY,
                     MCI_FROM | MCI_TO,
                     (DWORD)&mci_play))
    {
    m_private->Close();
    ppThrow (EAccessFailure, "Error playing track");
    }

  m_private->Close();
  }

//============================================================================

void AudioCD::Play(int track)
  {
  Play(track, track);
  }

//============================================================================

void AudioCD::Play()
  {
  Play(GetFirstTrack(), GetLastTrack());
  }

//============================================================================

void AudioCD::Stop()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }
  m_private->Open();
  mciSendCommand(m_private->m_mci_open.wDeviceID,MCI_STOP,0,0);
  m_private->Close();
  }

//============================================================================

void AudioCD::Pause()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }
  m_private->Open();
  mciSendCommand(m_private->m_mci_open.wDeviceID,MCI_PAUSE,0,0);
  m_private->Close();
  }

//============================================================================

void AudioCD::Resume()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }
  m_private->Open();
  mciSendCommand(m_private->m_mci_open.wDeviceID,MCI_RESUME,0,0);
  m_private->Close();
  }

//============================================================================

void AudioCD::Eject()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }
  m_private->Open();
  mciSendCommand(m_private->m_mci_open.wDeviceID,MCI_SET, MCI_SET_DOOR_OPEN, 0);
  m_private->Close();
  }

//============================================================================

void AudioCD::Retract()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return;
    }
  m_private->Open();
  mciSendCommand(m_private->m_mci_open.wDeviceID, MCI_SET, MCI_SET_DOOR_CLOSED, 0);
  m_private->Close();
  }

//============================================================================


int AudioCD::GetCurrentTrack()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return -1;
    }

  m_private->Open();
  int error;
  m_private->m_mci_status.dwTrack = 0;
  m_private->m_mci_status.dwCallback = 0;
  m_private->m_mci_status.dwItem = MCI_STATUS_CURRENT_TRACK;
  if ((error = mciSendCommand(m_private->m_mci_open.wDeviceID,
                              MCI_STATUS,
                              MCI_STATUS_ITEM | MCI_WAIT,
                              (DWORD)(LPVOID)&m_private->m_mci_status)))
    {
    char strError[128];
    mciGetErrorString(error, strError, 128);


    m_private->Close();
    ppThrow (EUnavailable, "Error getting the current position");
    }
  m_private->Close();
  return m_private->m_mci_status.dwReturn;
  }

//============================================================================

int AudioCD::GetFirstTrack()
  {
  return 1;
  }

//============================================================================

int AudioCD::GetLastTrack()
  {
  if (!m_private)
    {
    m_private = AudioCDPrivate::Create();
    if (!m_private)
      return -1;
    }
  return m_private->m_tracks;
  }

PP_I_NAMESPACE_END
PP_NAMESPACE_END

#endif /* defined(PP_SYS_OS_WIN32) */

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//============================================================================

AudioCD::AudioCD()
  : m_private(0)
{
  try
    {
      m_private = new AudioCDPrivate;
    }
  catch(...)
    {
    }
}

//============================================================================

AudioCD::~AudioCD()
{
  try
    {
      delete m_private;
    }
  catch(...)
    {
    }
}

//============================================================================

void AudioCD::Next()
{
  int current = GetCurrentTrack();
  int last = GetLastTrack();
  if (current != last)
    {
      Play(current+1, last);
    }
}

//============================================================================

void AudioCD::Previous()
{
  int current = GetCurrentTrack();
  int first = GetFirstTrack();
  if (current > first)
    {
      Play(current-1, GetLastTrack());
    }
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
