/* ====================================================================
 * Copyright (c) 1995-1997 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/* Module COCO, COrba COmmunicator
 * Derived from both the Apache and Mico examples.  This module is intended to 
 * show how to proceed from the client server architecture to a multi-tier
 * approach. This module is free under the conditions of the apache group.
 * As corba is defined in C++ we have to include all the C++ stuff separately.
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include <stdio.h>

#include "mod_coco.h"

static int coco_handler (request_rec *r) ;
static void coco_init (server_rec *s, pool *p) ;
static void *coco_dir_create (pool *p, char *dirspec) ;
static void *coco_dir_merge (pool *p, void *parent_conf, void *newloc_conf) ;
static void *coco_server_create (pool *p, server_rec *s) ;
static void *coco_server_merge (pool *p, void *server1_conf, void *server2_conf) ;
static int coco_xlate (request_rec *r) ;
static int coco_ckauth (request_rec *r) ;
static int coco_ckuser (request_rec *r) ;
static int coco_ckuser (request_rec *r) ;
static int coco_ckaccess (request_rec *r) ;
static int coco_typer (request_rec *r) ;
static int coco_fixer (request_rec *r) ;
static int coco_logger (request_rec *r) ;
static int coco_hparser (request_rec *r) ;

static const char *coco_handle_BIND
(cmd_parms *cmd, coco_config *mconfig, char *word1, char *word2){
  coco_corba_bind(mconfig,word1,word2);
  return NULL;
}

command_rec coco_commands[] = {
  { "CorbaBind", 
    coco_handle_BIND, 
    NULL, 
    RSRC_CONF, 
    TAKE2,
    "Arguments passed to the Corba initialisation" 
  },
  { NULL }
};

handler_rec coco_handlers[] = {
    { "coco-handler", coco_handler },
    { NULL }
};


/*
 * Let's set up a module-local static cell to point to the accreting callback
 * trace.  As each API callback is made to us, we'll tack on the particulars
 * to whatever we've already recorded.  To avoid massive memory bloat as
 * directories are walked again and again, we record the routine/environment
 * the first time (non-request context only), and ignore subsequent calls for
 * the same routine/environment.
 */
static char *trace = NULL;
static table *static_calls_made = NULL;

/*
 * To avoid leaking memory from pools other than the per-request one, we
 * allocate a module-private pool, and then use a sub-pool of that which gets
 * freed each time we modify the trace.  That way previous layers of trace
 * data don't get lost.
 */
static pool *coco_pool = NULL;
static pool *coco_subpool = NULL;

/*
 * Declare ourselves so the configuration routines can find and know us.
 * We'll fill it in at the end of the module.
 */

/*--------------------------------------------------------------------------*/
/*									    */
/* Finally, the list of callback routines and data structures that	    */
/* provide the hooks into our module from the other parts of the server.    */
/*									    */
/*--------------------------------------------------------------------------*/
/* 
 * Module definition for configuration.  If a particular callback is not
 * needed, replace its routine name below with the word NULL.
 *
 * The number in brackets indicates the order in which the routine is called
 * during request processing.  Note that not all routines are necessarily
 * called (such as if a resource doesn't have access restrictions).
 */
module coco_module = {
    STANDARD_MODULE_STUFF,
    coco_init,		/* initializer */
    coco_dir_create,		/* per-directory config creater */
    NULL,//coco_dir_merge,		/* dir config merger - default is to override */
    coco_server_create,	/* server config creator */
    NULL,//coco_server_merge,	/* server config merger */
    coco_commands,		/* command table */
    coco_handlers,		/* [6] list of handlers */
    coco_xlate,		/* [1] filename-to-URI translation */
    coco_ckuser,		/* [4] check/validate HTTP user_id */
    coco_ckauth,		/* [5] check HTTP user_id is valid *here* */
    coco_ckaccess,		/* [3] check access by host address, etc. */
    coco_typer,		/* [6] MIME type checker/setter */
    coco_fixer,		/* [7] fixups */
    coco_logger,		/* [9] logger */
    coco_hparser		/* [2] header parser */
};

/*
 * Locate our directory configuration record for the current request.
 */
static coco_config *our_dconfig
	(request_rec *r) {

    return (coco_config *) get_module_config
				(
				    r->per_dir_config,
				    &coco_module
				);
}

/*
 * Locate our server configuration record for the specified server.
 */
static coco_config *our_sconfig
	(server_rec *s) {

    return (coco_config *) get_module_config
				(
				    s->module_config,
				    &coco_module
				);
}

/*
 * Likewise for our configuration record for the specified request.
 */
static coco_config *our_rconfig
	(request_rec *r) {

    return (coco_config *) get_module_config
				(
				    r->request_config,
				    &coco_module
				);
}

/*
 * This routine sets up some module-wide cells if they haven't been already.
 */
static void setup_module_cells () {
    /*
     * If we haven't already allocated our module-private pool, do so now.
     */
    if (coco_pool == NULL) {
	coco_pool = make_sub_pool (NULL);
    };
    /*
     * Likewise for the table of routine/environment pairs we visit outside of
     * request context.
     */
    if (static_calls_made == NULL) {
	static_calls_made = make_table (coco_pool, 16);
    };
}

/*
 * This routine is used to add a trace of a callback to the list.  We're
 * passed the server record (if available), the request record (if available),
 * a pointer to our private configuration record (if available) for the
 * environment to which the callback is supposed to apply, and some text.  We
 * turn this into a textual representation and add it to the tail of the list.
 * The list can be displayed by the coco_handler() routine.
 *
 * If the call occurs within a request context (i.e., we're passed a request
 * record), we put the trace into the request pool and attach it to the
 * request via the notes mechanism.  Otherwise, the trace gets added
 * to the static (non-request-specific) list.
 *
 * Note that the r->notes table is only for storing strings; if you need to
 * maintain per-request data of any other type, you need to use another
 * mechanism.
 */

#define TRACE_NOTE "coco-trace"

static void trace_add
	(server_rec *s, request_rec *r, coco_config *mconfig,
	 const char *note) {

    char    *sofar;
    char    *addon;
    char    *where;
    pool    *p;
    char    *trace_copy;
    coco_config
	    *rconfig;

    /*
     * Make sure our pools and tables are set up - we need 'em.
     */
    setup_module_cells ();
    /*
     * Now, if we're in request-context, we use the request pool.
     */
    if (r != NULL) {
	p = r->pool;
	if ((trace_copy = table_get (r->notes, TRACE_NOTE)) == NULL) {
	    trace_copy = "";
	}
    } else {
	/*
	 * We're not in request context, so the trace gets attached to our
	 * module-wide pool.  We do the create/destroy every time we're called
	 * in non-request context; this avoids leaking memory in some of
	 * the subsequent calls that allocate memory only once (such as the
	 * key formation below).
	 *
	 * Make a new sub-pool and copy any existing trace to it.  Point the
	 * trace cell at the copied value.
	 */
	p = make_sub_pool (coco_pool);
	if (trace != NULL) {
	    trace = pstrdup (p, trace);
	}
	/*
	 * Now, if we have a sub-pool from before, nuke it and replace with
	 * the one we just allocated.
	 */
	if (coco_subpool != NULL) {
	    destroy_pool (coco_subpool);
	}
	coco_subpool = p;
	trace_copy = trace;
    }
    /*
     * If we weren't passed a configuration record, we can't figure out to
     * what location this call applies.  This only happens for co-routines
     * that don't operate in a particular directory or server context.  If we
     * got a valid record, extract the location (directory or server) to which
     * it applies.
     */
    where = (mconfig != NULL) ? mconfig->loc : "nowhere";
    where = (where != NULL) ? where : "";
    /*
     * Now, if we're not in request context, see if we've been called with
     * this particular combination before.  The table is allocated in the
     * module's private pool, which doesn't get destroyed.
     */
    if (r == NULL) {
	char	*key;

	key = pstrcat (p, note, ":", where, NULL);
	if (table_get (static_calls_made, key) != NULL) {
	    /*
	     * Been here, done this.
	     */
	    return;
	} else {
	    /*
	     * First time for this combination of routine and environment -
	     * log it so we don't do it again.
	     */
	    table_set (static_calls_made, key, "been here");
	}
    }
    addon = pstrcat 
		(
		    p,
		    "   <LI>\n",
		    "    <DL>\n",
		    "     <DT><SAMP>",
		    note,
		    "</SAMP>\n",
		    "     </DT>\n",
		    "     <DD><SAMP>[",
		    where,
		    "]</SAMP>\n",
		    "     </DD>\n",
		    "    </DL>\n",
		    "   </LI>\n",
		    NULL
		);
    sofar = (trace_copy == NULL) ? "" : trace_copy;
    trace_copy = pstrcat (p, sofar, addon, NULL);
    if (r != NULL) {
	table_set (r->notes, TRACE_NOTE, trace_copy);
    } else {
	trace = trace_copy;
    }
}


/*--------------------------------------------------------------------------*/
/*									    */
/* Now we declare our content handlers, which are invoked when the server   */
/* encounters a document which our module is supposed to have a chance to   */
/* see.  (See mod_mime's SetHandler and AddHandler directives, and the	    */
/* mod_info and mod_status examples, for more details.)			    */
/*									    */
/* Since content handlers are dumping data directly into the connexion	    */
/* (using the r*() routines, such as rputs() and rprintf()) without	    */
/* intervention by other parts of the server, they need to make		    */
/* sure any accumulated HTTP headers are sent first.  This is done by	    */
/* calling send_http_header().  Otherwise, no header will be sent at all,   */
/* and the output sent to the client will actually be HTTP-uncompliant.	    */
/*--------------------------------------------------------------------------*/
/* 
 * Sample content handler.  All this does is display the call list that has
 * been built up so far.
 *
 * The return value instructs the caller concerning what happened and what to
 * do next:
 *  OK ("we did our thing")
 *  DECLINED ("this isn't something with which we want to get involved")
 *  HTTP_mumble ("an error status should be reported")
 */
static int coco_handler(request_rec *r) {
  
  coco_config
    *dcfg;
  coco_config
    *rcfg;
  
  dcfg = our_dconfig (r);
  trace_add (r->server, r, dcfg, "coco_handler()");
  
/*
 * ready with the preparations. The doing is left to the C++ parts
 */
  
  return coco_corba_request(r,dcfg);
}


/* 
 * This function is called during server initialisation.  Any information
 * that needs to be recorded must be in static cells, since there's no
 * configuration record.
 *
 * There is no return value.
 */

static void coco_init
	(server_rec *s, pool *p) {

    char    *note;
    char    *sname = s->server_hostname;

    /*
     * Set up any module cells that ought to be initialised.
     */
    setup_module_cells ();
    /*
     * The arbitrary text we add to our trace entry indicates for which server
     * we're being called.
     */
    sname = (sname != NULL) ? sname : "";
    note = pstrcat (p, "coco_init(", sname, ")", NULL);
    trace_add (s, NULL, NULL, note);
}

/*
 * This function gets called to create up a per-directory configuration
 * record.  This will be called for the "default" server environment, and for
 * each directory for which the parser finds any of our directives applicable.
 * If a directory doesn't have any of our directives involved (i.e., they
 * aren't in the .htaccess file, or a <Location>, <Directory>, or related
 * block), this routine will *not* be called - the configuration for the
 * closest ancestor is used.
 *
 * The return value is a pointer to the created module-specific
 * structure.
 */
static void *coco_dir_create
	(pool *p, char *dirspec) {

    coco_config
	    *cfg;
    char    *dname = dirspec;

    /*
     * Allocate the space for our record from the pool supplied.
     */
    cfg = (coco_config *) pcalloc (p, sizeof(coco_config));
    /*
     * Now fill in the defaults.  If there are any `parent' configuration
     * records, they'll get merged as part of a separate callback.
     */
    cfg->local = 0;
    cfg->cmode = CONFIG_MODE_DIRECTORY;
    
    coco_corba_init(cfg);

   /*
     * Finally, add our trace to the callback list.
     */
    dname = (dname != NULL) ? dname : "";
    cfg->loc = pstrcat (p, "DIR(", dname, ")", NULL);
    trace_add (NULL, NULL, cfg, "coco_dir_create()");
    return (void *) cfg;
}

/*
 * This function gets called to create a per-server configuration
 * record.  It will always be called for the "default" server.
 *
 * The return value is a pointer to the created module-specific
 * structure.
 */
static void *coco_server_create
	(pool *p, server_rec *s) {

    coco_config
	    *cfg;
    char    *sname = s->server_hostname;

    /*
     * As with the coco_dir_create() reoutine, we allocate and fill in an
     * empty record.
     */
    cfg = (coco_config *) pcalloc (p, sizeof(coco_config));
    cfg->local = 0;
    cfg->cmode = CONFIG_MODE_SERVER;
    /*
     * Note that we were called in the trace list.
     */
    sname = (sname != NULL) ? sname : "";
    cfg->loc = pstrcat (p, "SVR(", sname, ")", NULL);
    trace_add (s, NULL, cfg, "coco_server_create()");
    return (void *) cfg;
}

/*
 * This routine gives our module an opportunity to translate the URI into an
 * actual filename.  If we don't do anything special, the server's default

 * rules (Alias directives and the like) will continue to be followed.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are called for this phase.
 */
static int coco_xlate
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    /*
     * We don't actually *do* anything here, except note the fact that we were
     * called.
     */
    trace_add (r->server, r, cfg, "coco_xlate()");
    return DECLINED;
}

/*
 * This routine is called to check the authentication information sent with
 * the request (such as looking up the user in a database and verifying that
 * the [encrypted] password sent matches the one in the database).
 *
 * The return value is OK, DECLINED, or some HTTP_mumble error (typically
 * HTTP_UNAUTHORIZED).  If we return OK, no other modules are given a chance
 * at the request during this phase.
 */
static int coco_ckuser
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    /*
     * Don't do anything except log the call.
     */
    trace_add (r->server, r, cfg, "coco_ckuser()");
    return DECLINED;
}

/*
 * This routine is called to check to see if the resource being requested
 * requires authorisation.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * other modules are called during this phase.
 *
 * If *all* modules return DECLINED, the request is aborted with a server
 * error.
 */
static int coco_ckauth
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    /*
     * Log the call and return OK, or access will be denied (even though we
     * didn't actually do anything).
     */
    trace_add (r->server, r, cfg, "coco_ckauth()");
    return OK;
}

/*
 * This routine is called to check for any module-specific restrictions placed
 * upon the requested resource.  (See the mod_access module for an example.)
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  All modules with an
 * handler for this phase are called regardless of whether their predecessors
 * return OK or DECLINED.  The first one to return any other status, however,
 * will abort the sequence (and the request) as usual.
 */
static int coco_ckaccess
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    trace_add (r->server, r, cfg, "coco_ckaccess()");
    return OK;
}

/*
 * This routine is called to determine and/or set the various document type
 * information bits, like Content-type (via r->content_type), language, et
 * cetera.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, no
 * further modules are given a chance at the request for this phase.
 */
static int coco_typer
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    /*
     * Log the call, but don't do anything else - and report truthfully that
     * we didn't do anything.
     */
    trace_add (r->server, r, cfg, "coco_typer()");
    return DECLINED;
}

/*
 * This routine is called to perform any module-specific fixing of header
 * fields, et cetera.  It is invoked just before any content-handler.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, the
 * server will still call any remaining modules with an handler for this
 * phase.
 */
static int coco_fixer
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    /*
     * Log the call and exit.
     */
    trace_add (r->server, r, cfg, "coco_fixer()");
    return OK;
}

/*
 * This routine is called to perform any module-specific logging activities
 * over and above the normal server things.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, any
 * remaining modules with an handler for this phase will still be called.
 */
static int coco_logger
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    trace_add (r->server, r, cfg, "coco_logger()");
    return DECLINED;
}

/*
 * This routine is called to give the module a chance to look at the request
 * headers and take any appropriate specific actions early in the processing
 * sequence.
 *
 * The return value is OK, DECLINED, or HTTP_mumble.  If we return OK, any
 * remaining modules with handlers for this phase will still be called.
 */
static int coco_hparser
	(request_rec *r) {

    coco_config
	    *cfg;

    cfg = our_dconfig (r);
    trace_add (r->server, r, cfg, "coco_hparser()");
    return DECLINED;
}


